unit DirNavigator01;

interface
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  StrUtils, Menus, Dialogs,  ShlObj, ShellApi,
  StdCtrls, ExtCtrls, Grids, CheckLst,
  //  
  UnitInputDirDlg, UnitStatDir;

// --------------------------------------------------------------------
//    DestDir  SrcDir
//      
//   (   )
function DestDirIsSubSrcDir (SrcDir, DestDir : string) : boolean;


// ====================================================================
// TDriveList ()
//
//    RefrechDriveList()   
//       DriveStringList.
//      property DriveChar,  
//    property DriveStringList.
// ====================================================================
// --------------------------------------------------------------------
type TDriveList = class(TObject)
private
    fStrList   : TStringList;
protected
    function GetDriveChar(Indx : integer) : char;
public
    constructor Create();
    procedure Free();
    procedure RefrechDriveList();
    property DriveStringList : TStringList read fStrList;
    property DriveChar[Indx : integer] : char read GetDriveChar;
end;

// ====================================================================
// TComboBoxDriveList
//
//     TComboBox  
//        .  
//       OnDrawItem
//     Items.   
//    ,    
//   .      
//     onClick  onDblClick. 
//   CDROM  USB    
//   Refresh.
// ====================================================================
type TComboBoxDriveList = class(TDriveList)
private
    fcmbBox  : TComboBox;
    procedure ReDrawItem(Control: TWinControl;  Index: Integer;
                         Rect: TRect; State: TOwnerDrawState);
public
    constructor Create(RqComboBox : TComboBox);
    procedure Free();
    procedure Refresh();
end;

// ====================================================================
// TListBoxDriveList
//
//     TListBox  
//        .  
//       OnDrawItem
//     Items.   
//    ,    
//   .      
//     onClick  onDblClick. 
//   CDROM  USB    
//   Refresh.
// ====================================================================
type TListBoxDriveList = class(TDriveList)
private
    fLstBox  : TListBox;
    procedure ReDrawItem(Control: TWinControl;  Index: Integer;
                         Rect: TRect; State: TOwnerDrawState);
public
    constructor Create(RqListBox : TListBox);
    procedure Free();
    procedure Refresh();
end;

// ====================================================================
// TIndicator
//
//       
// .     TImage 
//      Next().
//      NextStep, 
//    Next()    . 
//  NextStep  100.
//        
//     TChoiceFiles.
// ====================================================================
type TIndicator = class(TObject)
private
  fImg      : TImage;
  fFonColor : TColor;
  fColor    : TColor;
  fNextStep : Word;
  fCount    : integer;
  fX        : integer;
  fXE       : integer;
  fYE       : integer;
public
  constructor Create(RqImage : TImage);
  procedure Clear();
  procedure Next();
  property FonColor : TColor read fFonColor write fFonColor;
  property Color    : TColor read fColor write fColor;
  property NextStep : Word   read fNextStep write fNextStep;
end;

// ====================================================================
// TDirNavigator
//
//     TStringGrid  
//          (
// ChoiceDir) .     
//   ,   faAnyFile. 
//     ,  
//   ChoiceDir.    
//        onNavigate,
//    onDblClick.    
//    ,    ('..').
//       
//  OnChangeDir,     
//  ,       
//  ChoiceDir.
//
//   TDirNavigator   TComboBoxDriveList:
//
// procedure TForm1.ComboBox1Click(Sender: TObject);
// var DriveChar : char;
// begin
//      TComboBoxDriveList  TDirNavigator
//   if Assigned(ComboBoxDriveList1) and Assigned(DirNavigator1)
//   then  begin
//      //    ComboBox1  
//      if not ComboBox1.ItemIndex < 0
//      then begin
//         DriveChar := ComboBoxDriveList1.DriveChar[ComboBox1.ItemIndex];
//             
//         DirNavigator1.ChoiceDir := DriveChar + ':'
//      end;
//   end;
// end;
// ====================================================================
type TActionCode =(acNone, acCopy, acCut, acPaste, acDelete, acNotReadOnly);
type TDirNavigator = class(TObject)
private
    // ------------
    // 
    fGrid          : TStringGrid;
    // ------------
    //  
    fFileAttr      : word;         //    
    fChoiceDir     : string;       //    
    fOnChangeDir   : TNotifyEvent; // .  :  fChoiceDir
    // ------------
    fActionParm    : record
       fCmdCode    : TActionCode;  //   
       fSrcDir     : string;       //  
       fSelName    : string;       //   
       fSelType    : word;         //  fSelName : 0  faDirectory
       fDestDir    : string;       //  
       //  
       fNotCopyCount  : cardinal;  //   
       fNotMoveCount  : cardinal;  //   
       fNotDelCount   : cardinal;  //     
       fNotRdOnlCount : cardinal;  //    ReadOnly
    end;
    //   fChoiceDir
    fDirStat   : record
       fDirsCount  : cardinal;     //    
       fFilesCount : cardinal;     //    
       fFilesSize  : cardinal;     //    
    end;
    fIndicator     : TIndicator;   //   
    // ---------------------
    //   
    procedure PrepareActionParm(RqCmdCode : TActionCode);
    //    
    procedure MenuClick(Sender : TObject);
    //     fGrid  PopupMenu
    procedure GreateAndConnecPoUpMenu();
    // 
    procedure FreePopupMenu();
    // ---------------------
    procedure ClearShowDir();
    procedure ReShowDir(RqDir : string);
    procedure RunOnNavigate(Sender : TObject);
    procedure SetChoiceDir(RqDir : string);
    procedure SetfFileAttr(RqAttr : word);
    procedure GetDirStat(RqDir : string);
    // ---------------------
    //      
    procedure RunShellExecute(RqFileName : string);
    //    
    function OneFileCopy(SrcFileName, DestFileName : string) : boolean;
    //      
    procedure RunRqCopyOrMove();
    //    
    procedure RunRqDelete();
    //      ReadOnly
    procedure RunRqNotReadOnly();
    //  , ,   
    //  ReadOnly    
    procedure RunTreeDirCmd(SrcDirName, DestDirName : string;
                            RqAction : TActionCode);
public
    constructor Create(RqStringGrid : TStringGrid);
    procedure Free();
    property  FileAttr    : word read fFileAttr write SetfFileAttr;
    property  ChoiceDir   : string read fChoiceDir write SetChoiceDir;
    property  OnChangeDir : TNotifyEvent read fOnChangeDir write fOnChangeDir;
    //   
    property  Indicator   : TIndicator  read fIndicator write fIndicator;
end;

// ====================================================================
// TChoiceFiles
//
//     TCheckListBox  
//    ,    ( TreeDirs:=False)
//    ( TreeDirs:=True).  
//    ChoiceDir,  
//  .     
//  TonChoiceFile,       
// 'Name.Ext'.        
//  True.        
//  .  (  ChoiceDir)   
//     ( FileAttr). 
//   TonChoiceFile  .
// ====================================================================

type TonChoiceFile = function(ShortFileName : string) : boolean of object;

type TChoiceFiles = class(TObject)
private
    fCheckListBox : TCheckListBox;   //    TCheckListBox
    fFileAttr     : word;            //   
    fChoiceDir    : string;          //   
    fonChoiceFile : TonChoiceFile;   //    
    fTreeDirs     : boolean;         //    
    fCurrentIndx  : LongInt;         //   GetNextChecked()
    // --------------------------------
    //    
    procedure SetChoiceDir(RqDir : string);
    //     
    procedure SetfFileAttr(RqAttr : word);
    //     
    procedure GetFilesFromChoiceDir(RqDir : string);
    //      
    procedure GetFilesFromTreeDirs (RqDir : string);
public
    // ---------------------
    //  
    constructor Create(RqCheckListBox : TCheckListBox);
    procedure Free();
    //  
    procedure GetFilesToBox();
    //   
    property  FileAttr     : word read fFileAttr write SetfFileAttr;
    property  ChoiceDir    : string read fChoiceDir write SetChoiceDir;
    property  TreeDirs     : boolean read fTreeDirs write fTreeDirs;
    //     .
    //    ,    
    //     .
    property  onChoiceFile : TonChoiceFile read fonChoiceFile
                                          write fonChoiceFile;
    // ---------------------
    //  
    procedure SetAllChecked(RqCheck : boolean);
    function GetCheckedCount(RqCheck : boolean) : Longint;
    //      RqCheck
    //  ''     
    function GetFirstChecked(RqCheck : boolean;
                         var CommonPos : integer) : string;
    //      RqCheck
    //  ''      
    function GetNextChecked(RqCheck : boolean;
                        var CommonPos : integer) : string;
end;

// --------------------------------------------------------------------
//     TonChoiceFile
type TChoiceDat = record
    Ext        : string;    //     ( '.TXT')
    //  
    PrefEn     : boolean;   // True -    
    PrefUse    : byte;      // 0 - ,   0 -   
    PrefStr    : string;    //   (( 'SAV_')
    //  
    SufxEn     : boolean;   // True -    
    SufxUse    : byte;      // 0 - ,   0 -   
    SufxStr    : string;    //   ( '_SAV')
end;

// --------------------------------------------------------------------
//     TonChoiceFile (  HarryFan)
//           
function ChoiceFileFilter (ShortFileName  : string;
                           ChoiceDat : TChoiceDat) : boolean;

// ====================================================================
// ====================================================================

implementation
{$R DirNavigator01.res}

// ====================================================================
// ====================================================================



// ====================================================================
//    
// ====================================================================
// --------------------------------------------------------------------
type  TDriveType = (dtUnknown, dtNoDrive, dtFloppy, dtFixed,
                    dtNetwork, dtCDROM,  dtRAM);
// --------------
type TTextCase   = (tcLowerCase, tcUpperCase);
// --------------
//    Bitmap
// (  InitsDriveBitmaps())
var FloppyBMP, FixedBMP, NetworkBMP, CDROMBMP, RAMBMP: TBitmap;

// ====================================================================
//  
// ====================================================================
// --------------------------------------------------------------------
//     Bitmap
//  HInstance   UNIT SysInit,
//     uses UNIT System.
procedure InitsDriveBitmaps();
begin
  { assign bitmap glyphs }
  FloppyBMP := TBitmap.Create;
  FloppyBMP.Handle := LoadBitmap(HInstance, 'FLOPPY');
  FixedBMP := TBitmap.Create;
  FixedBMP.Handle := LoadBitmap(HInstance, 'HARD');
  NetworkBMP := TBitmap.Create;
  NetworkBMP.Handle := LoadBitmap(HInstance, 'NETWORK');
  CDROMBMP := TBitmap.Create;
  CDROMBMP.Handle := LoadBitmap(HInstance, 'CDROM');
  RAMBMP := TBitmap.Create;
  RAMBMP.Handle := LoadBitmap(HInstance, 'RAM');
end;
// --------------------------------------------------------------------
//      
function VolumeID(DriveChar: Char): string;
var
  OldErrorMode: Integer;
  NotUsed, VolFlags: DWORD;
  Buf: array [0..MAX_PATH] of Char;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    Buf[0] := #$00;
    if GetVolumeInformation(PChar(DriveChar + ':\'), Buf, DWORD(sizeof(Buf)),
      nil, NotUsed, VolFlags, nil, 0) then
      SetString(Result, Buf, StrLen(Buf))
    else Result := '';
    if DriveChar < 'a' then
      Result := AnsiUpperCaseFileName(Result)
    else
      Result := AnsiLowerCaseFileName(Result);
      Result := Format('[%s]',[Result]);
  finally
    SetErrorMode(OldErrorMode);
  end;
end;
// --------------------------------------------------------------------
//      
function NetworkVolume(DriveChar: Char): string;
var
  Buf: Array [0..MAX_PATH] of Char;
  DriveStr: array [0..3] of Char;
  BufferSize: DWORD;
begin
  BufferSize := sizeof(Buf);
  DriveStr[0] := UpCase(DriveChar);
  DriveStr[1] := ':';
  DriveStr[2] := #0;
  if WNetGetConnection(DriveStr, Buf, BufferSize) = WN_SUCCESS then
  begin
    SetString(Result, Buf, BufferSize);
    if DriveChar < 'a' then
      Result := AnsiUpperCaseFileName(Result)
    else
      Result := AnsiLowerCaseFileName(Result);
  end
  else
    Result := VolumeID(DriveChar);
end;
// --------------------------------------------------------------------
//     
procedure DriveBuildList(RqItems : TStrings);
var
  DriveNum: Integer;
  DriveChar: Char;
  DriveType: TDriveType;
  DriveBits: set of 0..25;

  procedure AddDrive(const VolName: string; Obj: TObject);
  begin
    RqItems.AddObject(Format('%s: %s',[DriveChar, VolName]), Obj);
  end;

begin
  RqItems.Clear;
  Integer(DriveBits) := GetLogicalDrives;
  for DriveNum := 0 to 25 do
  begin
    if not (DriveNum in DriveBits) then Continue;
    DriveChar := Char(DriveNum + Ord('a'));
    DriveType := TDriveType(GetDriveType(PChar(DriveChar + ':\')));
    DriveChar := Upcase(DriveChar);

    case DriveType of
      dtFloppy:   RqItems.AddObject(DriveChar + ':', FloppyBMP);
      dtFixed:    AddDrive(VolumeID(DriveChar), FixedBMP);
      dtNetwork:  AddDrive(NetworkVolume(DriveChar), NetworkBMP);
      dtCDROM:    AddDrive(VolumeID(DriveChar), CDROMBMP);
      dtRAM:      AddDrive(VolumeID(DriveChar), RAMBMP);
    end;
  end;
end;
// ====================================================================
//   TDriveList
// ====================================================================
// --------------------------------------------------------------------

constructor TDriveList.Create();
begin
   inherited Create;
   fStrList := TStringList.Create;
   InitsDriveBitmaps();
   RefrechDriveList();
end;
// --------------------------------------------------------------------
procedure TDriveList.Free();
begin
   fStrList.Free;
   inherited Free;
end;
// --------------------------------------------------------------------
//     
procedure TDriveList.RefrechDriveList();
var DriveNum: Integer;
    DriveChar: Char;
    DriveType: TDriveType;
    DriveBits: set of 0..25;
    // ---------------------
    procedure AddDrive(const VolName: string; Obj: TObject);
    begin
      fStrList.AddObject(Format('%s: %s',[DriveChar, VolName]), Obj);
    end;
    // ---------------------
begin
  fStrList.Clear;
  Integer(DriveBits) := GetLogicalDrives;
  for DriveNum := 0 to 25 do
  begin
    if not (DriveNum in DriveBits) then Continue;
    DriveChar := Char(DriveNum + Ord('a'));
    DriveType := TDriveType(GetDriveType(PChar(DriveChar + ':\')));
    DriveChar := Upcase(DriveChar);
    case DriveType of
      dtFloppy:   fStrList.AddObject(DriveChar + ': (Removable)', FloppyBMP);
      dtFixed:    AddDrive(VolumeID(DriveChar), FixedBMP);
      dtNetwork:  AddDrive(NetworkVolume(DriveChar), NetworkBMP);
      dtCDROM:    AddDrive(VolumeID(DriveChar), CDROMBMP);
      dtRAM:      AddDrive(VolumeID(DriveChar), RAMBMP);
      else fStrList.AddObject(DriveChar + ': (Unknown)', nil);
    end;
  end;
end;
//        
function TDriveList.GetDriveChar(Indx : integer) : char;
begin
   Result := #0;
   if (Indx >= 0) and (Indx < fStrList.Count)
   then Result := fStrList[Indx][1];
end;

// ====================================================================
// TListBoxDriveList
// ====================================================================
// --------------------------------------------------------------------
constructor TListBoxDriveList.Create(RqListBox : TListBox);
begin
  inherited Create;
  fLstBox := RqListBox;
  fLstBox.font.Name := 'Tahoma';
  fLstBox.font.Size := 8;
  fLstBox.Items := DriveStringList;
  fLstBox.Style := lbOwnerDrawVariable;
  fLstBox.OnDrawItem := ReDrawItem;
end;
// --------------------------------------------------------------------
procedure TListBoxDriveList.Free();
begin
  inherited Free;
end;
// --------------------------------------------------------------------
procedure TListBoxDriveList.ReDrawItem(Control: TWinControl;  Index: Integer;
                                       Rect: TRect; State: TOwnerDrawState);
const  Pos1 = 25;  //    
var    wBMP : TBitmap;
       wBmpW, wBmpH  : integer;
begin
  if Control is TListBox
  then begin
    with (Control as TListBox) do
    begin
      //  
      Canvas.Pen.Style   := psClear;
      Canvas.Brush.Style := bsSolid;
      Canvas.FillRect(Rect);
      wBmpW := 16;
      wBMP := TBitmap(Items.Objects[Index]);
      if wBMP <> nil
      then begin
         wBmpW := wBMP.Width;
         wBmpH := wBMP.Height;
         Canvas.BrushCopy(Bounds(Rect.Left + 2,
                                 (Rect.Top + Rect.Bottom - wBmpH) div 2,
                                 wBmpW, wBmpH),
                          // Source
                          wBMP, Bounds(0, 0, wBmpW, wBmpH),
                          // Color
                          wBMP.Canvas.Pixels[0, wBmpH - 1]);
      end;
      //  
      Canvas.TextOut(Rect.Left + wBmpW + 8, Rect.Top, Items.Strings[Index]);
    end;
  end;
end;
// --------------------------------------------------------------------
procedure TListBoxDriveList.Refresh();
begin
    RefrechDriveList();
    fLstBox.Items := DriveStringList;
end;

// ====================================================================
// TComboBoxDriveList
// ====================================================================
// --------------------------------------------------------------------
constructor TComboBoxDriveList.Create(RqComboBox : TComboBox);
begin
  inherited Create;
  fcmbBox := RqComboBox;
  fcmbBox.font.Name := 'Tahoma';
  fcmbBox.font.Size := 8;
  fcmbBox.Items := DriveStringList;
  fcmbBox.Style := csOwnerDrawVariable;
  fcmbBox.OnDrawItem := ReDrawItem;
end;
// --------------------------------------------------------------------
procedure TComboBoxDriveList.Free();
begin
  inherited Free;
end;
// --------------------------------------------------------------------
procedure TComboBoxDriveList.ReDrawItem(Control: TWinControl;  Index: Integer;
                                        Rect: TRect; State: TOwnerDrawState);
var    wBMP : TBitmap;
       wBmpW, wBmpH  : integer;
begin
  if Control is TComboBox
  then begin
    with (Control as TComboBox) do
    begin
      //  
      Canvas.Pen.Style   := psClear;
      Canvas.Brush.Style := bsSolid;
      Canvas.FillRect(Rect);
      wBmpW := 16;
      wBMP := TBitmap(Items.Objects[Index]);
      if wBMP <> nil
      then begin
         wBmpW := wBMP.Width;
         wBmpH := wBMP.Height;
         Canvas.BrushCopy(Bounds(Rect.Left + 2,
                                 (Rect.Top + Rect.Bottom - wBmpH) div 2,
                                 wBmpW, wBmpH),
                          // Source
                          wBMP, Bounds(0, 0, wBmpW, wBmpH),
                          // Color
                          wBMP.Canvas.Pixels[0, wBmpH - 1]);
      end;
      //  
      Canvas.TextOut(Rect.Left + wBmpW + 8, Rect.Top, Items.Strings[Index]);
    end;
  end;
end;

procedure TComboBoxDriveList.Refresh();
begin
    RefrechDriveList();
    fcmbBox.Items := DriveStringList;
end;

// ====================================================================
// TDirNavigator
// ====================================================================

// --------------------------------------------------------------------
constructor TDirNavigator.Create(RqStringGrid : TStringGrid);
begin
  inherited Create;
  fGrid := RqStringGrid;
  fGrid.FixedCols  := 1;
  fGrid.FixedRows  := 1;
  fGrid.DefaultRowHeight := 16;
  fGrid.ScrollBars := ssBoth;
  fGrid.Options := [goFixedVertLine,goFixedHorzLine,goVertLine,
                    goHorzLine,goColSizing,goRowSelect];
  fGrid.font.Name := 'Tahoma';
  fGrid.font.Size := 8;
  fGrid.OnDblClick   := RunOnNavigate;
  fOnChangeDir := nil;
  fIndicator := nil;

  GreateAndConnecPoUpMenu();
  PrepareActionParm(acNone);
  // -------------------
  fFileAttr  := faAnyFile;
  fChoiceDir := '';
  // -------------------
  ReShowDir(fChoiceDir);
end;
// --------------------------------------------------------------------
procedure TDirNavigator.Free();
begin
  FreePopupMenu();
  inherited Free;
end;
// --------------------------------------------------------------------
//     fGrid  PopupMenu
procedure  TDirNavigator.GreateAndConnecPoUpMenu();
var MenuItems : array of TMenuItem;
begin
   SetLength(MenuItems, 13);
   // --------------------
   MenuItems[0]:= NewItem('   ', TextToShortCut(''),
                  False, True, MenuClick, 0, 'M_StatDir');
   MenuItems[0].Tag := 1;
   // --------------------
   // 
   MenuItems[1]:= NewItem('-', TextToShortCut(''), False, True, nil, 0, '');
   MenuItems[1].Tag := 0;
   // --------------------
   MenuItems[2]:= NewItem(' (    )',
                  TextToShortCut(''), False, True, MenuClick, 0, 'M_Copy');
   MenuItems[2].Tag := 2;
   // --------------------
   MenuItems[3]:= NewItem(' (    )',
                  TextToShortCut(''), False, True, MenuClick, 0, 'M_Cut');
   MenuItems[3].Tag := 3;
   // --------------------
   // 
   MenuItems[4]:= NewItem('-', TextToShortCut(''), False, True, nil, 0, '');
   MenuItems[4].Tag := 0;
   // --------------------
   MenuItems[5]:= NewItem(' (      )',
                  TextToShortCut(''), False, True, MenuClick, 0, 'M_Paste');
   MenuItems[5].Tag := 4;
   // --------------------
   // 
   MenuItems[6]:= NewItem('-', TextToShortCut(''),  False, True, nil, 0, '');
   MenuItems[6].Tag := 0;
   // --------------------
   MenuItems[7]:= NewItem('    ',
                  TextToShortCut(''), False, True, MenuClick, 0, 'M_Delete');
   MenuItems[7].Tag := 5;
   // --------------------
   // 
   MenuItems[8]:= NewItem('-', TextToShortCut(''),  False, True, nil, 0, '');
   MenuItems[8].Tag := 0;
   // --------------------
   MenuItems[9]:= NewItem('   ReadOnly',
                  TextToShortCut(''), False, True, MenuClick, 0, 'M_NotRO');
   MenuItems[9].Tag := 6;
   // --------------------
   // 
   MenuItems[10]:= NewItem('-', TextToShortCut(''),  False, True, nil, 0, '');
   MenuItems[10].Tag := 0;
   // --------------------
   MenuItems[11]:= NewItem('  ', TextToShortCut(''),
                  False, True, MenuClick, 0, 'M_CreateDir');
   MenuItems[11].Tag := 7;
   // --------------------
   MenuItems[12]:= NewItem('  ', TextToShortCut(''),
                  False, True, MenuClick, 0, 'M_StatDir');
   MenuItems[12].Tag := 8;
   // ====================
   //    (PopupMenu)
   fGrid.PopupMenu := NewPopupMenu(fGrid, 'Menu',paLeft, True, MenuItems);
   // ====================
   SetLength(MenuItems,0);
end;
// --------------------------------------------------------------------
//  PopupMenu
procedure TDirNavigator.FreePopupMenu();
var indx : integer;
begin
   if not Assigned(fGrid.PopupMenu)then Exit;
   for indx := fGrid.PopupMenu.Items.Count - 1 downto 0
   do fGrid.PopupMenu.Items.Delete(indx);
   fGrid.PopupMenu.Destroy;
   fGrid.PopupMenu := nil;
end;
// --------------------------------------------------------------------
//  TDirNavigator.   
// --------------------------------------------------------------------
//      
function FAttrToStr (RqAttr : word) : string;
var WStr : string;
begin
  WStr := '';
  if (RqAttr and faSymLink)   > 0 then WStr := '[LNK] ';
  if (RqAttr and faDirectory) > 0 then WStr := WStr + '[DIR] ';
  if (RqAttr and faVolumeID)  > 0 then WStr := WStr + '[VID] ';
  if (RqAttr and faSysFile)   > 0 then WStr := WStr + 'S';
  if (RqAttr and faHidden)    > 0 then WStr := WStr + 'H';
  if (RqAttr and faReadOnly)  > 0 then WStr := WStr + 'R';
  if (RqAttr and faArchive)   > 0 then WStr := WStr + 'A';
  if WStr = '' then WStr := '?';
  Result := WStr;
end;

// --------------------------------------------------------------------
//    
function GetParentDir(RqDir : string) : string;
var wLen, indx : integer;
begin
    Result := RqDir;
    wLen := Length(RqDir);
    indx := wLen;
    while indx > 2   //     ,  'C:'
    do begin
        if RqDir[indx] = '\'
        then begin
           Result := LeftStr(Result, indx - 1);
           Exit;
        end;
        indx := indx - 1;
    end;
end;
// --------------------------------------------------------------------
//    fGrid
procedure TDirNavigator.ClearShowDir();
const TitArr : array [0..4] of string =
      ('',' ','',' ()','   ');
var   indx : integer;
begin
    with fGrid
    do begin
      RowCount  := 1;
      ColCount  := 5;
      for indx := 0 to ColCount - 1 do Cells[indx, 0] := TitArr[indx];
    end;
end;
// --------------------------------------------------------------------
//   
procedure TDirNavigator.ReShowDir(RqDir : string);
var   sr         : TSearchRec;
      CodeEnd    : integer;
      Ind        : integer;
begin
  ClearShowDir();
  if fChoiceDir = '' then Exit;
  if not DirectoryExists(fChoiceDir) then Exit;
  with fGrid do
  begin
    CodeEnd := FindFirst(RqDir + '\*.*', fFileAttr, sr);
    if CodeEnd = 0
    then begin
      repeat
        sr.Attr := sr.Attr and $ff;
        if ((sr.Attr and fFileAttr) = sr.Attr) and (sr.Name <> '')
        then begin
          Ind := RowCount;
          RowCount := RowCount + 1;
          Cells[0,Ind] := IntToStr(Ind);
          Cells[1,Ind] := sr.Name;
          if (sr.Attr and faDirectory) > 0
          then Cells[3,Ind] := ''  //   
          else Cells[3,Ind] := IntToStr(sr.Size);
          Cells[4,Ind] := DateTimeToStr(FileDateToDateTime(sr.Time));
          if (sr.Attr and faDirectory) > 0
          then Objects [2, Ind] := Pointer(faDirectory)
          else Objects [2, Ind] := nil;
          if (Trim(sr.Name) = '.')
          then  Cells[2,Ind] := ' SELF'
          else  if (Trim(sr.Name) = '..')
                then Cells[2,Ind] := ' RETURN'
                else Cells[2,Ind] := FAttrToStr(sr.Attr);
        end;
      until FindNext(sr) <> 0;
      FindClose(sr);
    end
    else begin
       //  CodeEnd = 2  ShowMessage(SysErrorMessage(CodeEnd))
       //  : '    '
       // ,       .
       //       RqDir  :

       if not (Length(RqDir) = 2)
       then begin
          //      
          Ind := RowCount;
          RowCount := RowCount + 1;
          Cells[0,Ind] := IntToStr(Ind);
          Cells[1,Ind] := '..';
          Cells[2,Ind] := ' RETURN';
          Objects [2, Ind] := Pointer(faDirectory);
          Cells[3,Ind] := '';
          Cells[4,Ind] := '';
          MessageDlg('   .' + #13#10
                + '   RETURN.',
                 mtInformation, [mbOk], 0);
       end
       else RowCount := 1;
    end;
  end; // of with fGrids
  //   
  if Assigned(fOnChangeDir) then fOnChangeDir(Self);
end;
// --------------------------------------------------------------------
//    
procedure TDirNavigator.RunOnNavigate(Sender : TObject);
begin
  with fGrid do
  begin
     if (Objects[2, Row] <> nil)
     then begin
        if (Cells[1,Row] = '..')
        then fChoiceDir := GetParentDir(fChoiceDir)
        else fChoiceDir := fChoiceDir + '\' + Cells[1,Row];
        ReShowDir(fChoiceDir);
     end;
  end;
end;
// --------------------------------------------------------------------
//          
procedure  TDirNavigator.SetfFileAttr(RqAttr : word);
begin
  fFileAttr := RqAttr and $0000007F;
  if fChoiceDir = '' then Exit;
  ReShowDir(fChoiceDir);
end;
// --------------------------------------------------------------------
//     (  property)
procedure TDirNavigator.SetChoiceDir(RqDir : string);
begin
   if Length(RqDir) < 2
   then Exit
   else begin
       if RqDir[Length(RqDir)] = '\'
       then fChoiceDir := LeftStr(RqDir, Length(RqDir)-1)
       else fChoiceDir := RqDir;
       ReShowDir(fChoiceDir);
   end;
end;
// --------------------------------------------------------------------
procedure TDirNavigator.GetDirStat(RqDir : string);
var  wDir      : string;
     sr        : TSearchRec;
begin
   if FindFirst(RqDir + '\*.*', faAnyFile, sr) = 0
   then begin
      repeat
      // -----------------------
        if (sr.Name <> '')
        then begin
           if (sr.Name <> '.') and (sr.Name <> '..')
           then begin
              if (sr.Attr and faDirectory) > 0
              then begin
                 Inc(fDirStat.fDirsCount);
                 wDir := RqDir + '\' + sr.Name;
                 GetDirStat(wDir);
              end
              else begin
                 Inc(fDirStat.fFilesCount);
                 fDirStat.fFilesSize := fDirStat.fFilesSize + sr.Size;
              end;
           end;
        end;
      // -----------------------
      if Assigned(fIndicator) then fIndicator.Next;
      until (FindNext(sr) <> 0);
      FindClose(sr);
   end;
end;
// --------------------------------------------------------------------
//  TDirNavigator.   ,  PopUpMenu
// --------------------------------------------------------------------
// --------------------------------------------------------------------
//      
// uses ShlObj, ShellApi;
procedure TDirNavigator.RunShellExecute(RqFileName : string);
var wFileName  : string;
    wRetutnCode : HINST;
    wMsg       : string;
begin
 wFileName := Trim(RqFileName);
 if not FileExists(wFileName) then Exit;
 //     
 wMsg := '';
 wRetutnCode := ShellExecute(HWND(0), nil, PChar(wFileName),
                            nil, nil, Sw_ShowNormal);
 case wRetutnCode of
 0	:
   wMsg := '       .';
 ERROR_FILE_NOT_FOUND :
   wMsg := '   .';
 ERROR_PATH_NOT_FOUND	:
   wMsg := '   .';
 ERROR_BAD_FORMAT	:
   wMsg := '  .exe ( Win32 .exe     .exe).';
 SE_ERR_ACCESSDENIED :
   wMsg := '      .';
 SE_ERR_ASSOCINCOMPLETE	:
   wMsg := '     .';
 SE_ERR_DDEBUSY :
  wMsg := '    DDE,'
        + '      DDE.';
 SE_ERR_DDEFAIL	:
  wMsg := '  DDE.';
 SE_ERR_DDETIMEOUT  :
  wMsg := ' DDE    ,'
        + '     .';
 SE_ERR_DLLNOTFOUND :
   wMsg := ' DLL  .';
 SE_ERR_NOASSOC	:
   wMsg := ' ,      .';
 SE_ERR_OOM :
   wMsg := '    .';
 SE_ERR_SHARE	 : wMsg :='  .';
 end;
 if wMsg <> '' then MessageDlg(wMsg, mtWarning, [mbOk], 0);
end;

// --------------------------------------------------------------------
//    
function TDirNavigator.OneFileCopy(SrcFileName, DestFileName : string): boolean;
var wFileName, wDestName : string;
    SrcFile  : TFileStream;
    DestFile : TFileStream;
begin
   Result := False;
   if not FileExists(SrcFileName) then Exit;
   if (UpperCase(SrcFileName) = UpperCase(DestFileName)) or
      (DestFileName = '')
   then begin
      //       
      wFileName := ExtractFileName(SrcFileName);
      wDestName := ExtractFileDir(SrcFileName) + '\' + ' ' + wFileName;
   end
   else wDestName := DestFileName;
   //  
   SrcFile  := nil;
   DestFile := nil;
   SrcFile := TFileStream.Create(SrcFileName, fmOpenRead or fmShareDenyWrite);
   try
      DestFile := TFileStream.Create(wDestName, fmCreate or fmShareDenyRead);
      try
        DestFile.CopyFrom(SrcFile, SrcFile.Size);
        Result := True;
      except end;
   except end;
   if Assigned(DestFile) then FreeAndNil(DestFile);
   if Assigned(SrcFile)  then FreeAndNil(SrcFile);
end;
// --------------------------------------------------------------------




// --------------------------------------------------------------------
//      ( '.'  '..'  )
function GetDirCount(Dir : string) : cardinal;
var sr     : TSearchRec;
begin
   Result := 0;
   if FindFirst(Dir + '\*.*', faAnyFile, sr) = 0 then
   if (sr.Name <> '')
   then begin
      repeat
         if not((sr.Name <> '.') or (sr.Name <> '..'))
         then Inc(Result);
      until FindNext(sr) <> 0;
   end;
   FindClose(sr);
end;
// --------------------------------------------------------------------
//  , ,   
//  ReadOnly    
procedure TDirNavigator.RunTreeDirCmd(SrcDirName, DestDirName : string;
                                      RqAction : TActionCode);
var  wSrcDir,  wDestDir  : string;
     sr : TSearchRec;
     wSrcFile, wDestFile : string;
     wDirCount : cardinal;
     wFileAttr : word;
begin
   // ----------------------------------------
   if FindFirst(SrcDirName + '\*.*', fFileAttr, sr) = 0
   then begin
      // -----------------------
      repeat
      // -----------------------
        if (sr.Name <> '')
        then begin
           if (sr.Name <> '.') and (sr.Name <> '..')
           then begin
              if (sr.Attr and faDirectory) > 0
              then begin
                 //  
                 wSrcDir  := SrcDirName  + '\' + sr.Name;
                 wDestDir := DestDirName + '\' + sr.Name;
                 RunTreeDirCmd(wSrcDir, wDestDir, RqAction);
              end
              else begin
                 //     
                 wSrcFile  := SrcDirName  + '\' + sr.Name;
                 wDestFile := DestDirName + '\' + sr.Name;
                 case RqAction of
                   // -----------
                   acCopy : begin
                              try
                                 //   
                                 if not DirectoryExists(DestDirName)
                                 then begin
                                   if not ForceDirectories(DestDirName)
                                   then begin
                                     Inc(fActionParm.fNotCopyCount);
                                     Break;
                                   end;
                                 end;
                                 if not DirectoryExists(DestDirName)
                                 then begin
                                   Inc(fActionParm.fNotCopyCount);
                                   Break;
                                 end;
                                 if not OneFileCopy(wSrcFile, wDestFile)
                                 then begin
                                   Inc(fActionParm.fNotCopyCount);
                                   Break;
                                 end;
                              except
                                 Inc(fActionParm.fNotCopyCount);
                                 Break;
                              end;
                            end;
                   // -----------
                   acCut : begin  // 
                              try
                                 //   
                                 if not DirectoryExists(DestDirName)
                                 then begin
                                   if not ForceDirectories(DestDirName)
                                   then begin
                                     Inc(fActionParm.fNotCopyCount);
                                     Break;
                                   end;
                                 end;
                                 if not DirectoryExists(DestDirName)
                                 then begin
                                   Inc(fActionParm.fNotCopyCount);
                                   Break;
                                 end;
                                 if not OneFileCopy(wSrcFile, wDestFile)
                                 then begin
                                   Inc(fActionParm.fNotCopyCount);
                                   Break;
                                 end;
                                 //    
                                 //     
                                 if not DeleteFile(wSrcFile)
                                 then begin
                                    Inc(fActionParm.fNotMoveCount);
                                    Break;
                                 end;
                              except
                                 //     
                                 Inc(fActionParm.fNotMoveCount);
                                 Break;
                              end;
                           end;
                   // -----------
                   acDelete :
                           begin
                                //   
                                if not DeleteFile(wSrcFile)
                                then begin
                                   Inc(fActionParm.fNotDelCount);
                                   Break;
                                end;
                              end;
                   // -----------
                   acNotReadOnly :
                           begin
                                //   ReadOnly   
                                wFileAttr := FileGetAttr(wSrcFile);
                                wFileAttr := wFileAttr and (not faReadOnly);
                                try
                                    FileSetAttr(wSrcFile, wFileAttr);
                                except end;
                                wFileAttr := FileGetAttr(wSrcFile);
                                if (wFileAttr and  faReadOnly) > 0
                                then begin
                                    Inc(fActionParm.fNotRdOnlCount);
                                    Break;
                                end;
                            end;
                   // -----------
                 end; // of case
              end; // of if (sr.Attr and faDirectory) > 0
           end; // of if (sr.Name <> '.') and (sr.Name <> '..')
        end; // of if (sr.Name <> '')

        if Assigned(fIndicator)
        then fIndicator.Next;

      // -----------------------
      until (FindNext(sr) <> 0);
      // -----------------------
      FindClose(sr);
      // ----------------------------------------
      //     
      with fActionParm do
      begin
          if ((RqAction = acDelete) and (fNotDelCount  = 0)) or
             ((RqAction = acCut)    and (fNotMoveCount = 0))
          then begin
             //      
             wDirCount := GetDirCount(SrcDirName);
             if wDirCount = 0
             then if not RemoveDir(SrcDirName)
                  then  fNotDelCount := fNotDelCount + wDirCount
             else fNotDelCount := fNotDelCount + wDirCount;
          end;
      end;
   end; // of if FindFirst(SrcDirName + '\*.*', fFileAttr, sr) = 0
end;

// --------------------------------------------------------------------
//   
procedure TDirNavigator.PrepareActionParm(RqCmdCode : TActionCode);
begin
   with fActionParm
   do begin
      //  
      fNotCopyCount  := 0;  //   
      fNotMoveCount  := 0;  //   
      fNotDelCount   := 0;  //     
      fNotRdOnlCount := 0;  //    ReadOnly
      case RqCmdCode of
      acNone :   begin
                    fCmdCode := acNone;
                    fSrcDir  := '';
                    fSelName := '';
                    fSelType := 0;
                    fDestDir := '';
                 end;
      acCopy, acCut :
                 begin
                    if fGrid.Row < 1
                    then begin
                       fCmdCode := acNone;
                       MessageDlg(' ' + #13#10
                                + '   ',
                                   mtInformation, [mbOk], 0);
                       Exit;
                    end;
                    //   
                    fCmdCode := RqCmdCode;
                    fSrcDir  := fChoiceDir;
                    fSelName := fGrid.Cells[1, fGrid.Row];
                    fSelType := word(fGrid.Objects[2, fGrid.Row]);
                    fDestDir := '';
                 end;
      acPaste :  begin
                    //   
                    //  fCmdCode   
                    // acCopy  acCut
                    if (fCmdCode = acCopy) or (fCmdCode = acCut)
                    then fDestDir := fChoiceDir
                    else fDestDir := '';
                 end;
      acDelete : begin
                    if fGrid.Row < 1
                    then begin
                       fCmdCode := acNone;
                       MessageDlg(' ' + #13#10
                                + '   ',
                                   mtInformation, [mbOk], 0);
                       Exit;
                    end;
                    //   
                    fCmdCode := acDelete;
                    fSrcDir  := fChoiceDir;
                    fSelName := fGrid.Cells[1, fGrid.Row];
                    fSelType := word(fGrid.Objects[2, fGrid.Row]);
                    fDestDir := '';
                end;
      acNotReadOnly :
                begin
                    if fGrid.Row < 1
                    then begin
                       fCmdCode := acNone;
                       MessageDlg(' ' + #13#10
                                + '   ',
                                   mtInformation, [mbOk], 0);
                       Exit;
                    end;
                    //     ReadOnly
                    fCmdCode := acNotReadOnly;
                    fSrcDir  := fChoiceDir;
                    fSelName := fGrid.Cells[1, fGrid.Row];
                    fSelType := word(fGrid.Objects[2, fGrid.Row]);
                    fDestDir := '';
                end;
      end; // of case
   end;
end;

// --------------------------------------------------------------------
//    DestDir  SrcDir
//      
//   (   )
function DestDirIsSubSrcDir (SrcDir, DestDir : string) : boolean;
var wStr : string;
begin
   Result := False;
   if UpCase(SrcDir[1]) <> UpCase(DestDir[1]) then Exit;
   if Length(DestDir) >=  Length(SrcDir)
   then begin
      wStr := LeftStr(DestDir, Length(SrcDir));
      if UpperCase(wStr) = UpperCase(SrcDir)
      then Result := True;
   end;
end;
// --------------------------------------------------------------------
//      
procedure TDirNavigator.RunRqCopyOrMove();
var wFileAttrSave : word;
    wSrcDir       : string;
    wDestDir      : string;
    wSrcFileName  : string;
    wDestFileName : string;
    wStr          : string;
begin
    with fActionParm do
    begin
       if not((fCmdCode = acCopy) or (fCmdCode = acCut))  then Exit;
       //   
       wFileAttrSave := fFileAttr;
       fFileAttr := faAnyFile;    // /   
       if fSelType =  faDirectory
       then begin
           //    
           if (fSrcDir = '') or (fDestDir = '') or (fDestDir = fSrcDir)
           then Exit;
           wSrcDir  := fSrcDir  + '\' + fSelName;
           wDestDir := fDestDir + '\' + fSelName;
           //    
           if DestDirIsSubSrcDir (wSrcDir, wDestDir)
           then begin
               MessageDlg('    '
               + #13#10 + '   !'
               + #13#10 + ' : ' + wSrcDir
               + #13#10 + ' : ' + wDestDir
               + #13#10 + '  ... ',
                mtWarning, [mbOk], 0);
                Exit;
           end;
           wStr := '';
           if fCmdCode = acCopy
           then wStr := '   !';
           if fCmdCode = acCut
           then wStr := '   !';
           if wStr = '' then Exit;
           if MessageDlg(wStr
             + #13#10 + ' : ' + wSrcDir
             + #13#10 + ' : ' + wDestDir
             + #13#10  + '?:',
               mtConfirmation, [mbYes, mbNo], 0) = mrYes
           then RunTreeDirCmd(wSrcDir, wDestDir, fActionParm.fCmdCode);
           // 
           if (fNotCopyCount = 0) and
              (fNotMoveCount = 0) and
              (fNotDelCount  = 0)
           then MessageDlg('  ',
                            mtInformation, [mbOk], 0)
           else MessageDlg('    !'
                + #13#10 + '    ,'
                + #13#10 + '    ReadOnly.',
                  mtWarning, [mbOk], 0);
       end
       else begin
           //    
           wSrcFileName  := fSrcDir + '\' + fSelName;
           if (fCmdCode = acCopy)
           then begin
              //   
              if (fDestDir = '') or (not DirectoryExists(fDestDir))
              then wDestFileName := wSrcFileName
              else wDestFileName := fDestDir + '\' + fSelName;
              //    
              //        
              if not OneFileCopy(wSrcFileName, wDestFileName)
              then Inc(fActionParm.fNotCopyCount);
           end;
           if (fCmdCode = acCut)
           then begin
              if DirectoryExists(fDestDir)
              then begin
                  //     
                  wDestFileName := fDestDir + '\' + fSelName;
                  if OneFileCopy(wSrcFileName, wDestFileName)
                  then begin
                      //     
                      //     
                      if not DeleteFile(wSrcFileName)
                      then Inc(fActionParm.fNotMoveCount);
                  end
                  else Inc(fActionParm.fNotMoveCount);
              end
              else Inc(fActionParm.fNotMoveCount);
           end;
       end;
       ReShowDir(fChoiceDir);
       //    
       fFileAttr := wFileAttrSave;
    end;
end;
// --------------------------------------------------------------------
//    
procedure TDirNavigator.RunRqDelete();
var wFileAttrSave : word;
    wSrcDir       : string;
    wSrcFileName  : string;
begin
    with fActionParm do
    begin
       if (fCmdCode <> acDelete) then Exit;
       //   
       wFileAttrSave := fFileAttr;
       fFileAttr := faAnyFile;      //    
       if fSelType =  faDirectory
       then begin
            //   
            wSrcDir  := fSrcDir  + '\' + fSelName;
            if MessageDlg('    :'
              + #13#10  + wSrcDir
              + #13#10  + '?:',
               mtConfirmation, [mbYes, mbNo], 0) = mrYes
            then RunTreeDirCmd(wSrcDir, '', acDelete);
            // 
            if (fNotDelCount  = 0)
            then MessageDlg('  ',
                             mtInformation, [mbOk], 0)
            else MessageDlg('    !'
                 + #13#10 + '    ,'
                 + #13#10 + '    ReadOnly.',
                  mtWarning, [mbOk], 0);
       end
       else begin
          //   
          wSrcFileName  := fSrcDir + '\' + fSelName;
          if not DeleteFile(wSrcFileName)
          then begin
                Inc(fActionParm.fNotDelCount);
                MessageDlg('    : ' + #13#10
                          + wSrcFileName,  mtWarning, [mbOk], 0);
          end;
       end;
       ReShowDir(fChoiceDir);
       //    
       fFileAttr := wFileAttrSave;
    end;
end;
// --------------------------------------------------------------------
//      ReadOnly
procedure TDirNavigator.RunRqNotReadOnly();
var wFileAttrSave : word;
    wFileAttr     : word;
    wSrcDir       : string;
    wSrcFileName  : string;
begin
    with fActionParm do
    begin
       if (fCmdCode <> acNotReadOnly) then Exit;
       //   
       wFileAttrSave := fFileAttr;
       fFileAttr := faAnyFile;     //    
       if fSelType =  faDirectory
       then begin
            //   ReadOnly    
            wSrcDir  := fSrcDir  + '\' + fSelName;
            if MessageDlg('    ReadOnly'
              + #13#10  + '      :'
              + #13#10  + wSrcDir
              + #13#10  + '?:',
               mtConfirmation, [mbYes, mbNo], 0) = mrYes
            then RunTreeDirCmd(wSrcDir, '', acNotReadOnly);
            // 
            if (fNotDelCount  = 0)
            then MessageDlg('  ',
                             mtInformation, [mbOk], 0)
            else MessageDlg('    !'
                 + #13#10 + '    .',
                  mtWarning, [mbOk], 0);
       end
       else begin
          //   ReadOnly  
          wSrcFileName  := fSrcDir + '\' + fSelName;
          wFileAttr := FileGetAttr(wSrcFileName);
          wFileAttr := wFileAttr and (not faReadOnly);
          try
              FileSetAttr(wSrcFileName, wFileAttr);
          except end;
          wFileAttr := FileGetAttr(wSrcFileName);
          if (wFileAttr and  faReadOnly) > 0
          then begin
                Inc(fActionParm.fNotRdOnlCount);
                MessageDlg('   ReadOnly c  : '
                 + #13#10 + wSrcFileName,
                   mtWarning, [mbOk], 0);
          end;
       end;
       ReShowDir(fChoiceDir);
       //    
       fFileAttr := wFileAttrSave;
    end;
end;
// --------------------------------------------------------------------
//   integer  ,  
function NumStrFormat (RqSrcStr : string) : string;
var wSrcStr, wStr : string;
begin
    Result := '';
    wSrcStr := RqSrcStr;
    while Length(wSrcStr) > 2
    do begin
      wStr := RightStr(wSrcStr, 3);
      Result := wStr + ' ' + Result;
      Delete(wSrcStr, Length(wSrcStr) - 2, 3)
    end;
    Result := wSrcStr + ' ' + Result;
end;

// --------------------------------------------------------------------
//    
procedure TDirNavigator.MenuClick(Sender : TObject);
var wItem : TMenuItem;
    wStr  : string;
    wChar : char;
    wDrive: Byte;
    wDiskFree : Int64;
begin
   if fChoiceDir = '' then Exit;
   wChar :=UpCase(fChoiceDir[1]);
   if Assigned(fIndicator) then
   begin
      if (wChar = 'A') or (wChar = 'B')
      then fIndicator.NextStep := 1
      else fIndicator.NextStep := 100;
      fIndicator.Clear;
   end;
   wItem := TMenuItem(Sender);
   with fActionParm do
   begin
      case wItem.Tag of
      1 : begin
             //    
             if fGrid.Row < 1 then Exit;
             if fChoiceDir = '' then Exit;
             if Word(fGrid.Objects[2, fGrid.Row])= faDirectory
             then Exit;
             wStr := fChoiceDir + '\' + fGrid.Cells[1, fGrid.Row];
             RunShellExecute(wStr);
          end;
      2 : begin
             //   
             PrepareActionParm(acCopy);
          end;
      3 : begin
             //   
             PrepareActionParm(acCut);
          end;
      4 : begin
             //      
             PrepareActionParm(acPaste);
             if Assigned(fIndicator) then fIndicator.Clear;
             RunRqCopyOrMove();
             if Assigned(fIndicator) then fIndicator.Clear;
          end;
      5 : begin
             //    
             PrepareActionParm(acDelete);
             if Assigned(fIndicator) then fIndicator.Clear;
             RunRqDelete();
             if Assigned(fIndicator) then fIndicator.Clear;
          end;
      6 : begin
             //   ReadOnly
             PrepareActionParm(acNotReadOnly);
             //      ReadOnly
             if Assigned(fIndicator) then fIndicator.Clear;
             RunRqNotReadOnly();
             if Assigned(fIndicator) then fIndicator.Clear;
          end;
      7 : begin
              if fChoiceDir = '' then Exit;
              //  
              InputDirDlg.ShowModal;
              if InputDirDlg.ModalResult = mrOk
              then begin
                 wStr := InputDirDlg.edDirName.Text;
                 if wStr = '' then Exit;
                 wStr := fChoiceDir + '\' + wStr;
                 if not DirectoryExists(wStr)
                 then ForceDirectories(wStr);
                 ReShowDir(fChoiceDir);
              end;
          end;
      8 : begin
             if fChoiceDir = '' then Exit;
             //  
             FillChar(fDirStat, SizeOF(fDirStat), #0);
             //    
             wChar :=UpCase(fChoiceDir[1]);
             if Assigned(fIndicator) then fIndicator.Clear;
             //  
             GetDirStat(fChoiceDir);
             if Assigned(fIndicator) then fIndicator.Clear;
             with FormStatDir do
             begin
               wDrive := Ord(wChar) - Ord('A') + 1;
               if wDrive >= 1
               then begin
                   wDiskFree := DiskFree(wDrive);
                   wStr := NumStrFormat(IntToStr(wDiskFree));
                   stxtDiskFree.Caption := wStr + ' ';
               end;
               stxtDirNane.Caption := fChoiceDir;
               stxtDirCount.Caption   := IntToStr(fDirStat.fDirsCount);
               stxtFilesCount.Caption := IntToStr(fDirStat.fFilesCount);
               wStr := NumStrFormat(IntToStr(fDirStat.fFilesSize));
               stxtFilesSize.Caption  := wStr + ' ';
               ShowModal;
             end;
          end;
      end; // of case
   end; // of with
end;

// ====================================================================
// TIndicator
// ====================================================================
// --------------------------------------------------------------------
constructor TIndicator.Create(RqImage : TImage);
begin
   inherited Create;
   fImg      := RqImage;
   fFonColor := clBtnFace;
   fColor    := RGB(100,255,100);
   fNextStep := 100;
   fXE       := fImg.Width;
   fYE       := fImg.Height;
   Clear();
end;
// --------------------------------------------------------------------
//  
procedure TIndicator.Clear();
begin
   fX      := 0;
   fCount  := 0;
   fImg.Canvas.Pen.Color := fColor;
   fImg.Canvas.Pen.Style := psSolid;
   fImg.Canvas.Brush.Color := fFonColor;
   fImg.Canvas.Brush.Style := bsSolid;
   fImg.Canvas.FillRect(Rect(0,0,fImg.Width, fImg.Height));
end;
// --------------------------------------------------------------------
procedure TIndicator.Next();
begin
   Inc(fCount);
   if fCount < fNextStep
   then Exit;
   with fImg.Canvas do
   begin
      MoveTo(fX, 0);
      LineTo(fX, fYE);
      Inc(fX);
      if fX > fXE
      then begin
          fX := 0;
          FillRect(Rect(0,0,fImg.Width, fImg.Height));
      end;
   end;
   fCount := 0;
   fImg.Repaint;
end;

// ====================================================================
//   TChoiceFiles
// ====================================================================
// --------------------------------------------------------------------
//   
// --------------------------------------------------------------------
constructor TChoiceFiles.Create(RqCheckListBox : TCheckListBox);
begin
  inherited Create;
  fCheckListBox := RqCheckListBox;
  fFileAttr     := $0000007F;
  fChoiceDir    := GetCurrentDir();
  fonChoiceFile := nil;
  fTreeDirs     := False;
end;
// --------------------------------------------------------------------
procedure TChoiceFiles.Free();
begin
inherited Free;
end;
// --------------------------------------------------------------------
//    ( )
procedure TChoiceFiles.SetChoiceDir(RqDir : string);
begin
   if DirectoryExists(RqDir)
   then fChoiceDir := RqDir
   else begin
       fChoiceDir := '';
       MessageDlg('     .' + #13#10
                + '     .',
                 mtWarning, [mbOk], 0);
   end;
end;
// --------------------------------------------------------------------
//          
procedure  TChoiceFiles.SetfFileAttr(RqAttr : word);
begin
  fFileAttr := RqAttr and $0000007F;
  if fChoiceDir = '' then Exit;
  self.GetFilesToBox();
end;
// --------------------------------------------------------------------
//     
procedure TChoiceFiles.GetFilesFromChoiceDir(RqDir : string);
var  wDir   : string;
     sr     : TSearchRec;
begin
    wDir   := RqDir + '\*.*';
    if FindFirst(wDir, fFileAttr, sr) = 0 then
    begin
      repeat
        if (sr.Name <> '') and ((sr.Attr and faDirectory) = 0)
        then begin
          // ------------
          //     
          if not Assigned(fonChoiceFile)
          then begin
             //       
             //       
             //    (fChoiceDir)
             fCheckListBox.AddItem(RqDir + '\' + sr.Name,
                                   pointer(Length(fChoiceDir)));
          end
          else begin
             //     
             if fonChoiceFile(sr.Name)
             then begin
                //       
                //       
                //    (fChoiceDir)
                fCheckListBox.AddItem(RqDir + '\' + sr.Name,
                                             pointer(Length(fChoiceDir)));
             end;
          end;
      end;
      until FindNext(sr) <> 0;
      FindClose(sr);
    end;
end;
// --------------------------------------------------------------------
//      
procedure TChoiceFiles.GetFilesFromTreeDirs(RqDir : string);
var  wDir   : string;
     sr     : TSearchRec;
begin
    wDir   := RqDir + '\*.*';
    if FindFirst(wDir, fFileAttr, sr) = 0 then
    begin
      repeat
        if (sr.Name <> '')
        then begin
          if (sr.Name <> '.') and (sr.Name <> '..')
          then begin
             if ((sr.Attr and faDirectory) > 0)
             then begin
                //  
                WDir := RqDir + '\' + sr.Name;
                GetFilesFromTreeDirs(wDir);
             end
             else begin
                // ------------
                //     
                if not Assigned(fonChoiceFile)
                then begin
                   //       
                   //       
                   //    (fChoiceDir)
                   fCheckListBox.AddItem(RqDir + '\' + sr.Name,
                                         pointer(Length(fChoiceDir)));
                end
                else begin
                   //     
                   if fonChoiceFile(sr.Name)
                   then begin
                       //       
                       //       
                       //    (fChoiceDir)
                       fCheckListBox.AddItem(RqDir + '\' + sr.Name,
                                             pointer(Length(fChoiceDir)));
                   end;
                end;
                // ------------
             end;
          end;
      end;
      until FindNext(sr) <> 0;
      FindClose(sr);
    end;
end;
// --------------------------------------------------------------------
//   
procedure TChoiceFiles.GetFilesToBox();
begin
  fCheckListBox.Clear;
  if not DirectoryExists(fChoiceDir) then Exit;
  if fTreeDirs
  then GetFilesFromTreeDirs(fChoiceDir)
  else GetFilesFromChoiceDir(fChoiceDir);
end;
// --------------------------------------------------------------------
//   
// --------------------------------------------------------------------
//  Checked   
procedure TChoiceFiles.SetAllChecked(RqCheck : boolean);
var indx : integer;
begin
  for indx := 0 to fCheckListBox.Count - 1
  do fCheckListBox.Checked[indx] := RqCheck;
end;
// --------------------------------------------------------------------
//        
function TChoiceFiles.GetCheckedCount(RqCheck : boolean) : Longint;
var indx : integer;
begin
  Result := 0;
  for indx := 0 to fCheckListBox.Count - 1
  do if fCheckListBox.Checked[indx] = RqCheck
     then Inc(Result);
end;
// --------------------------------------------------------------------
//      RqCheck
//  ''     
function TChoiceFiles.GetFirstChecked(RqCheck : boolean;
                                  var CommonPos : integer) : string;
var indx : integer;
begin
  //      RqCheck   
  fCurrentIndx := -1;
  Result := '';
  for indx := 0 to fCheckListBox.Count - 1
  do if fCheckListBox.Checked[indx] = RqCheck
     then begin
        Result := fCheckListBox.Items.Strings[indx];
        CommonPos := integer(fCheckListBox.Items.Objects[indx]);
        //       RqCheck
        fCurrentIndx := indx;
        Exit;
     end;
end;
// --------------------------------------------------------------------
//      RqCheck
//  ''      
function TChoiceFiles.GetNextChecked(RqCheck : boolean;
                                 var CommonPos : integer) : string;
var indx : integer;
begin
   Result := '';
   if fCurrentIndx < 0
   //  GetFirstChecked      RqCheck
   then Exit;
   if not (fCurrentIndx < fCheckListBox.Count - 1)
   //     
   then begin
      // GetNextChecked     
      fCurrentIndx := -1;
      Exit;
   end;
   //    + 1   
   for indx := fCurrentIndx + 1 to fCheckListBox.Count - 1
   do begin
     if fCheckListBox.Checked[indx] = RqCheck
     then begin
        fCurrentIndx := indx;
        Result := fCheckListBox.Items.Strings[indx];
        CommonPos := integer(fCheckListBox.Items.Objects[indx]);
        Exit;
     end;
   end;
   // GetNextChecked     
   fCurrentIndx := -1;
end;

// ====================================================================
//     TonChoiceFile (  HarryFan)
// ====================================================================
// --------------------------------------------------------------------
//           
function ChoiceFileFilter (ShortFileName  : string;
                           ChoiceDat : TChoiceDat) : boolean;
var wExt       : string;
    wNameNE    : string;
    wName      : string;
    wEnPrefix  : boolean;
    wEnSufix   : boolean;
    wEnP, wEnS : boolean;
begin
   Result := False;
   wExt := UpperCase(ExtractFileExt(ShortFileName));
   if (wExt = ChoiceDat.Ext) or (ChoiceDat.Ext = '.*') or (ChoiceDat.Ext = '')
   then begin
      Result := True;
      // ---------------------------
      // ---------------------------
      if ChoiceDat.PrefEn or ChoiceDat.SufxEn
      //   
      then begin
          //     
          wNameNE := LeftStr(ShortFileName, Length(ShortFileName)-Length(wExt));
          // ---------------------------
          //    
          if ChoiceDat.PrefEn
          then begin
             wEnPrefix := False;
             if Length(wNameNE) > Length(ChoiceDat.PrefStr)
             then begin
                //   
                wName := LeftStr(wNameNE, Length(ChoiceDat.PrefStr));
                if UpperCase(wName) = ChoiceDat.PrefStr
                then wEnPrefix := True;
             end;
          end;
          // ---------------------------
          //    
          if ChoiceDat.SufxEn
          then begin
             wEnSufix := False;
             if Length(wNameNE) > Length(ChoiceDat.SufxStr)
             then begin
                //   
                wName := RightStr(wNameNE, Length(ChoiceDat.SufxStr));
                if UpperCase(wName) = ChoiceDat.SufxStr
                then wEnSufix := True;
             end;
          end;
          // ---------------------------
          //    
          if ChoiceDat.PrefEn and (not ChoiceDat.SufxEn)
          then if ChoiceDat.PrefUse = 0
               then Result := wEnPrefix
               else Result := not wEnPrefix;
          //    
          if (not ChoiceDat.PrefEn) and ChoiceDat.SufxEn
          then if ChoiceDat.SufxUse = 0
               then Result := wEnSufix
               else Result := not wEnSufix;
          //     
          if ChoiceDat.PrefEn and ChoiceDat.SufxEn
          then begin
               if ChoiceDat.PrefUse = 0
               then wEnP := wEnPrefix
               else wEnP := not wEnPrefix;
               if ChoiceDat.SufxUse = 0
               then wEnS := wEnSufix
               else wEnS := not wEnSufix;
               Result := wEnP and wEnS;
          end;
      end;  // of ChoiceDat.PrefEn or ChoiceDat.SufxEn
      // ---------------------------
      // ---------------------------
   end
   else Exit;
end;






// ====================================================================
//  
// ====================================================================
(*    . FileCtrl procedure TDriveComboBox
procedure TForm2.ComboBox1MeasureItem(Control: TWinControl; Index: Integer;
  var Height: Integer);
begin
  if Control is TComboBox
  then begin
    with (Control as TComboBox) do
    begin
       Height := 13;
    end;
  end;
end;
*)
// ====================================================================
// 
// ====================================================================


end.
